/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.daffodil.api.debugger;

import org.apache.daffodil.runtime1.debugger.DaffodilDebugger;

/**
 * Daffodil Debugger Runner interface to run the built-in Daffodil debugger.
 */
public interface DaffodilDebuggerRunner {
  /**
   * initialize using an Daffodil Debugger
   *
   * @param dd Daffodil debugger
   */
  void init(DaffodilDebugger dd);
  
  /**
   * Called by Daffodil when there is a pause in parsing to determine what
   * debugger actions should be taken.
   *
   * @return a debugger command that tells the Daffodil debugger what step to
   * take next.
   * @see <a target="_blank" href='https://daffodil.apache.org/debugger/'>Daffodil Debugger</a> - debugger commands
   */
  String getCommand();

  /**
   * Called by Daffodil when a debugger command has produced output. This method
   * is called once for every line produced by the Daffodil debugger.
   *
   * @param line a single line of output generated by the Daffodil debugger
   */
  void lineOutput(String line);

  /**
   * Called once upon completion of a parse, allowing one to perform any
   * steps necessary to clean up the debugger runner.
   */
  void fini();
}
