/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.drools.drl.parser.antlr4;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Collection;
import java.util.Map;
import java.util.Map.Entry;

import org.antlr.v4.runtime.FailedPredicateException;
import org.antlr.v4.runtime.InputMismatchException;
import org.antlr.v4.runtime.NoViableAltException;
import org.antlr.v4.runtime.RecognitionException;
import org.antlr.v4.runtime.Token;
import org.drools.drl.parser.DRLFactory;
import org.drools.drl.parser.DroolsParserException;
import org.drools.drl.parser.lang.DroolsParaphraseTypes;
import org.kie.internal.builder.conf.LanguageLevelOption;

/**
 * Helper class that generates DroolsParserException with user friendly error
 * messages.
 *
 * @see DroolsParserException
 */
public class DroolsParserExceptionFactory {
    public final static String                        TRAILING_SEMI_COLON_NOT_ALLOWED_MESSAGE = "Line %1$d:%2$d trailing semi-colon not allowed%3$s";
    public final static String                        PARSER_LOCATION_MESSAGE_COMPLETE        = " in %1$s %2$s";
    public final static String                        PARSER_LOCATION_MESSAGE_PART            = " in %1$s";
    public final static String                        UNEXPECTED_EXCEPTION                    = "Line %1$d:%2$d unexpected exception at input '%3$s'. Exception: %4$s. Stack trace:\n %5$s";

    private final Collection<Map<DroolsParaphraseTypes, String>> paraphrases;

    // TODO: need to deal with this array
    private String[]                                  tokenNames                              = null;

    private final LanguageLevelOption languageLevel;

    /**
     * DroolsParserErrorMessages constructor.
     *
     * @param tokenNames
     *            tokenNames generated by ANTLR
     * @param paraphrases
     *            paraphrases parser structure
     */
    public DroolsParserExceptionFactory(Collection<Map<DroolsParaphraseTypes, String>> paraphrases,
            LanguageLevelOption languageLevel) {
        this.paraphrases = paraphrases;
        this.languageLevel = languageLevel;
    }

    /**
     * This method creates a DroolsParserException for trailing semicolon
     * exception, full of information.
     *
     * @param line
     *            line number
     * @param column
     *            column position
     * @param offset
     *            char offset
     * @return DroolsParserException filled.
     */
    public DroolsParserException createTrailingSemicolonException( int line,
                                                                   int column,
                                                                   int offset ) {
        String message = String
                .format(
                         TRAILING_SEMI_COLON_NOT_ALLOWED_MESSAGE,
                         line,
                         column,
                         formatParserLocation() );

        return new DroolsParserException( "ERR 104",
                                          message,
                                          line,
                                          column,
                                          offset,
                                          null );
    }

    /**
     * This method creates a DroolsParserException full of information.
     *
     * @param offendingSymbol
     * @param line
     * @param charPositionInLine
     * @param e original exception
     * @return DroolsParserException filled.
     */
    public DroolsParserException createDroolsException(Object offendingSymbol, int line, int charPositionInLine, String message, RecognitionException e ) {
        return new DroolsParserException( determineErrorCode( e ),
                                          String.format("Line %d:%d %s", line, charPositionInLine, message),
                                          line,
                                          charPositionInLine,
                                          determineStartingIndex(offendingSymbol),
                                          e );
    }

    private int determineStartingIndex(Object offendingSymbol) {
        if (offendingSymbol instanceof Token) {
            return ((Token) offendingSymbol).getStartIndex();
        }
        return -1;
    }

    /**
     * Determines the error code based on a specific subtype of RecognitionException.
     */
    private String determineErrorCode( RecognitionException e ) {
        if (e == null) {
            return "ERR 109";
        }
        if ( e instanceof InputMismatchException) {
            return "ERR 102";
        }
        if ( e instanceof NoViableAltException ) {
            return "ERR 101";
        }
        if ( e instanceof FailedPredicateException ) {
            return "ERR 103";
        }
        throw new IllegalArgumentException("Unexpected exception type: " + e);
    }

    public DroolsParserException createDroolsException( Exception e,
                                                        Token token ) {
        StringWriter sw = new StringWriter();
        e.printStackTrace( new PrintWriter(sw) );
        return new DroolsParserException( String.format(
                                                         DroolsParserExceptionFactory.UNEXPECTED_EXCEPTION,
                                                         token.getLine(),
                                                         token.getCharPositionInLine(),
                                                         getBetterToken( token ),
                                                         e.toString(),
                                                         sw.toString() ),
                                          e );

    }

    /**
     * This will take Paraphrases stack, and create a sensible location
     */
    private String formatParserLocation() {
        StringBuilder sb = new StringBuilder();
        if ( paraphrases != null ) {
            for ( Map<DroolsParaphraseTypes, String> map : paraphrases ) {
                for ( Entry<DroolsParaphraseTypes, String> activeEntry : map.entrySet() ) {
                    if ( activeEntry.getValue().length() == 0 ) {
                        String kStr = getLocationName( activeEntry.getKey() );
                        if( kStr.length() > 0 ){
                            sb.append( String.format( PARSER_LOCATION_MESSAGE_PART, kStr ) );
                        }
                    } else {
                        sb.append( String.format( PARSER_LOCATION_MESSAGE_COMPLETE,
                                                  getLocationName( activeEntry.getKey() ),
                                                  activeEntry.getValue() ) );
                    }
                }
            }
        }
        return sb.toString();
    }

    /**
     * Returns a string based on Paraphrase Type
     *
     * @param type
     *            Paraphrase Type
     * @return a string representing the
     */
    private String getLocationName( DroolsParaphraseTypes type ) {
        switch ( type ) {
            case PACKAGE :
                return "package";
            case IMPORT :
                return "import";
            case FUNCTION_IMPORT :
                return "function import";
            case ACCUMULATE_IMPORT :
                return "accumulate import";
            case GLOBAL :
                return "global";
            case FUNCTION :
                return "function";
            case QUERY :
                return "query";
            case TEMPLATE :
                return "template";
            case RULE :
                return "rule";
            case RULE_ATTRIBUTE :
                return "rule attribute";
            case PATTERN :
                return "pattern";
            case EVAL :
                return "eval";
            default :
                return "";
        }
    }

    /**
     * Helper method that creates a user friendly token definition
     *
     * @param token
     *            token
     * @return user friendly token definition
     */
    private String getBetterToken( Token token ) {
        if ( token == null ) {
            return "";
        }
        return DRLFactory.getBetterToken(token.getType(), token.getText(), languageLevel);
    }
}
