/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import { MiningField, Model, PMML, Scorecard } from "@kie-tools/pmml-editor-marshaller";
import { Actions, AllActions } from "@kie-tools/pmml-editor/dist/editor/reducers";
import { Reducer } from "react";
import { HistoryService } from "@kie-tools/pmml-editor/dist/editor/history";
import { MiningSchemaFieldReducer } from "@kie-tools/pmml-editor/dist/editor/reducers/MiningSchemaFieldReducer";
import { ValidationRegistry } from "@kie-tools/pmml-editor/dist/editor/validation";

const historyService = new HistoryService([]);
const validationRegistry = new ValidationRegistry();
const miningFields: MiningField[] = [{ name: "field1" }];
const models: Model[] = [
  new Scorecard({
    MiningSchema: { MiningField: miningFields },
    functionName: "regression",
    Characteristics: { Characteristic: [] },
  }),
];
const pmml: PMML = {
  version: "1.0",
  DataDictionary: { DataField: [{ name: "field1", dataType: "boolean", optype: "categorical" }] },
  Header: {},
  models: models,
};
const reducer: Reducer<MiningField[], AllActions> = MiningSchemaFieldReducer(historyService, validationRegistry);

describe("MiningSchemaFieldReducer::Valid actions", () => {
  test("Actions.UpdateDataDictionaryField", () => {
    reducer(miningFields, {
      type: Actions.UpdateDataDictionaryField,
      payload: {
        modelIndex: 0,
        dataDictionaryIndex: 0,
        dataField: { name: "updated", dataType: "string", optype: "ordinal" },
        originalName: "field1",
      },
    });
    const updated = historyService.commit(pmml)?.models as Model[];
    const miningSchema = (updated[0] as Scorecard).MiningSchema;

    expect(miningSchema.MiningField).not.toEqual(miningFields);
    expect(miningSchema.MiningField.length).toBe(1);
    expect(miningSchema.MiningField[0].name).toBe("updated");
  });

  test("Actions.UpdateMiningSchemaField", () => {
    reducer(miningFields, {
      type: Actions.UpdateMiningSchemaField,
      payload: {
        modelIndex: 0,
        miningSchemaIndex: 0,
        name: "updated",
        originalName: "field1",
        usageType: "active",
        optype: "ordinal",
        importance: 5,
        outliers: "asExtremeValues",
        lowValue: -100,
        highValue: 120,
        missingValueTreatment: "asMean",
        missingValueReplacement: "a",
        invalidValueTreatment: "asValue",
        invalidValueReplacement: "b",
      },
    });
    const updated = historyService.commit(pmml)?.models as Model[];
    const miningSchema = (updated[0] as Scorecard).MiningSchema;

    expect(miningSchema.MiningField[0]).not.toEqual(miningFields[0]);
    expect(miningSchema.MiningField[0].name).toBe("updated");
    expect(miningSchema.MiningField[0].usageType).toBe("active");
    expect(miningSchema.MiningField[0].optype).toBe("ordinal");
    expect(miningSchema.MiningField[0].importance).toBe(5);
    expect(miningSchema.MiningField[0].outliers).toBe("asExtremeValues");
    expect(miningSchema.MiningField[0].lowValue).toBe(-100);
    expect(miningSchema.MiningField[0].highValue).toBe(120);
    expect(miningSchema.MiningField[0].missingValueTreatment).toBe("asMean");
    expect(miningSchema.MiningField[0].missingValueReplacement).toBe("a");
    expect(miningSchema.MiningField[0].invalidValueTreatment).toBe("asValue");
    expect(miningSchema.MiningField[0].invalidValueReplacement).toBe("b");
  });
});

describe("DataFieldReducer::Invalid actions", () => {
  test("Actions.SetHeaderDescription", () => {
    const updated: MiningField[] = reducer(miningFields, {
      type: Actions.SetHeaderDescription,
      payload: {
        description: "description",
      },
    });
    expect(updated).toEqual(miningFields);
  });
});
